﻿using log4net;
using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.WebServiceClient;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using VA.PPMS.Context;
using VA.PPMS.Context.Interface;
using VA.PPMS.IWS.Common;
using VA.PPMS.IWS.PersistenceService.Interface;
using VA.PPMS.ProviderData;

namespace VA.PPMS.IWS.PersistenceService
{
    public class PersistenceService : IPersistenceService
    {
        private readonly ILog _logger;
        private readonly IPpmsContextHelper _ppmsContextHelper;
        private readonly IPpmsHelper _ppmsHelper;

        public PersistenceService(ILog logger, IPpmsContextHelper ppmsContextHelper, IPpmsHelper ppmsHelper)
        {
            _logger = logger;
            _ppmsContextHelper = ppmsContextHelper;
            _ppmsHelper = ppmsHelper;
        }

        public async Task<MapperResult> ModifyAccountsAsync(MapperResult result, string transactionId)
        {
            try
            {
                _logger.Info($"@@@@ INFO - Start ModifyAccountsAsync for TransactionId: {transactionId} @@@@");

                if (result?.Details != null)
                {
                    foreach (var item in result.Details)
                    {
                        if (!item.IsValid) continue;

                        try
                        {
                            switch (item.Action)
                            {
                                case MapperAction.Insert:
                                    await InsertEntity(item.Provider);
                                    break;
                                case MapperAction.Update:
                                    //await UpdateEntity(item.Provider);
                                    await UpdateProvider(item.Provider);
                                    break;
                                case MapperAction.Delete:
                                    await DeleteEntity(item.Requests);
                                    break;
                            }
                        }
                        catch (Exception ex)
                        {
                            _logger.Error($"@@@@ ERROR - Exception ModifyAccountsAsync for TransactionId: {transactionId}; Message: {1} @@@@", ex);
                            item.Provider = null;
                            item.ValidationMessage = $"An exception occurred committing provider information: {ex.Message}";
                        }
                    }
                }

                _logger.Info($"@@@@ INFO - End ModifyAccountsAsync for TransactionId: {transactionId} @@@@");
                return result;
            }
            catch (Exception ex)
            {
                _logger.Error($"@@@@ ERROR - There was a problem Saving Add/Update Providers for TransactionId: {transactionId}.", ex);
                throw new PpmsServiceException($"There was a problem Saving Add/Update Providers for TransactionId: {transactionId}", ex);
            }
        }

        public async Task DeleteAccountsAsync(MapperResult result, string transactionId)
        {
            try
            {
                _logger.Info($"@@@@ INFO - Start DeleteAccountsAsync for TransactionId: {transactionId} @@@@");
                if (result?.Details != null)
                {
                    await DeleteAccounts(result.Details);
                }
                _logger.Info($"@@@@ INFO - End DeleteAccountsAsync for TransactionId: {transactionId} @@@@");
            }
            catch (Exception ex)
            {
               _logger.Error($"@@@@ ERROR - There was a problem Saving Deleted Providers for TransactionId: {transactionId}.", ex);
                throw new PpmsServiceException($"There was a problem Saving Deleted Providers for TransactionId: {transactionId}", ex);
            }
        }

        public async Task SaveBatchAsync(List<Validation> validationResults, DasMessage message)
        {
            var batchReferenceId = message.TransactionId;
            var networkId = await _ppmsHelper.GetNetworkIdByShorthand(message.SenderId);
            const int chunkSize = 100;

            try
            {
                if (validationResults != null && validationResults.Count > 0 && (!string.IsNullOrEmpty(batchReferenceId)) && (networkId.HasValue))
                {
                    using (var context = await _ppmsContextHelper.GetContextAsync())
                    {
                        var network = context.ppms_vaprovidernetworkSet.FirstOrDefault(x => x.Id == networkId.Value);
                        if (network == null) throw new PpmsServiceException("Error writing Batch record for Schema Validation. Cannot find NetworkId for the given Provider");

                        _logger.Info($"@@@@ INFO - Determining batch for TransactionId: {batchReferenceId} @@@@");
                        
                        // Check for existing batch record
                        var batch = await _ppmsHelper.GetBatch(message);
                        SaveChangesResultCollection saveChanges;
                        int batchStatus;
                        if (batch == null)
                        {
                            _logger.Info($"@@@@ DEBUG - Batch found for TransactionId: {batchReferenceId} @@@@");
                            // batch does not exist, create
                            batch = new ppms_batch
                            {
                                ppms_transactionid = message.TransactionId,
                                ppms_conversationid = message.ConversationId,
                                ppms_vaprovidernetwork_batch_network = network,
                                StatusCode = new OptionSetValue((int)ppms_batch_StatusCode.DataReceived)
                            };
                            context.AddObject(batch);

                            batchStatus = batch.StatusCode.Value;

                            // create batch record, if it doesn't exist
                            saveChanges = context.SaveChanges();
                            CheckSaveChanges(saveChanges);
                            batch = null;
                        }
                        else
                        {
                            _logger.Info($"@@@@ DEBUG - Batch NOT found for TransactionId: {batchReferenceId} @@@@");
                            batchStatus = batch.StatusCode.Value;
                        }

                        _logger.Info($"@@@@ INFO - Processing batch details for TransactionId: {batchReferenceId} @@@@");
                        var size = validationResults.Count;
                        var i = 0;

                        // process validation results, create batch detail records
                        foreach (var validationResult in validationResults)
                        {
                            try
                            {
                                i++;

                                // add batch to current context
                                if (batch == null)
                                {
                                    batch = await _ppmsHelper.GetBatch(message);
                                    context.Attach(batch);
                                }

                                if (batch == null)
                                {
                                    _logger.Error($"@@@@ ERROR - Unable to establish Batch Records for TransactionId: {batchReferenceId} @@@@");
                                    return;
                                }

                                var transactionType = GetTransactionType(validationResult.TransactionType);

                                var batchDetail = new ppms_batchdetail
                                {
                                    ppms_name = validationResult.ProviderName,
                                    ppms_isvalid = validationResult.IsValid,
                                    ppms_providerid = validationResult.ProviderId,
                                    ppms_transactiontype = new OptionSetValue(transactionType),
                                    ppms_provider = string.IsNullOrEmpty(validationResult.CorrelationId) ? null : new EntityReference(Account.EntityLogicalName, new Guid(validationResult.CorrelationId))
                                };
                                context.AddRelatedObject(batch, new Relationship("ppms_batch_batchdetail_batch"), batchDetail);

                                var batchDetailResult = new ppms_batchdetailresult
                                {
                                    ppms_isvalid = validationResult.IsValid,
                                    ppms_name = validationResult.ProviderId,
                                    ppms_entitytype = validationResult.ErrorSource,
                                    ppms_result = validationResult.IsValid
                                        ? "Your provider file record has been successfully verified and updated in the Provider Profile Management System (PPMS)."
                                        : "Your provider file record is invalid. Please correct as indicated and resubmit the provider the information has been corrected.",
                                    ppms_message = validationResult.Result,
                                };
                                context.AddRelatedObject(batchDetail, new Relationship("ppms_batchdetail_batchdetailresult"), batchDetailResult);

                                if (i % chunkSize == 0 || i == size)
                                {
                                    _logger.DebugFormat("--- DEBUG - Saving batch detail {0}-{1} of {2}. [TransactionId: {3}, ProviderId: {4}] ----", Math.Max(1, i - chunkSize), i, size, batchReferenceId, validationResult.ProviderId);

                                    saveChanges = context.SaveChanges();
                                    CheckSaveChanges(saveChanges);
                                    batch = null;
                                }
                            }
                            catch (Exception ex)
                            {
                                _logger.Error($"@@@@ ERROR - There was a problem Saving Batch Records for TransactionId: {batchReferenceId}, ProviderId: {validationResult.ProviderId} @@@@", ex);
                            }
                        }

                        // change batch status once processing is complete
                        if (batchStatus == (int)ppms_batch_StatusCode.DataReceived)
                        {
                            await _ppmsHelper.UpdateBatch(message, "Provider import process complete", (int)ppms_batch_StatusCode.Processing);
                        }
                    }
                }
                else if (!networkId.HasValue)
                {
                    _logger.Error($"@@@@ ERROR - There was a problem Saving Batch Records for TransactionId: {batchReferenceId}. Network ID ({message.SenderId}) is invalid. @@@@");
                }
            }
            catch (Exception ex)
            {
                _logger.Error($"@@@@ ERROR - There was an error Saving Batch Records for TransactionId: {batchReferenceId} @@@@", ex);
                throw new PpmsServiceException($"There was a problem Saving Batch Records for TransactionId: {batchReferenceId}", ex);
            }
        }

        private static void CheckSaveChanges(SaveChangesResultCollection saveChanges)
        {
            if (!saveChanges.HasError) return;

            var sb = new StringBuilder();
            foreach (var saveChange in saveChanges)
            {
                if (saveChange.Error != null)
                {
                    sb.AppendFormat("Error: {0}\n", saveChange.Error.Message);
                }
            }
            throw new PpmsServiceException($"Failure occurred while saving Batch records: {sb}");
        }

        private static int GetTransactionType(int transactionType)
        {
            switch (transactionType)
            {
                case (int)TransactionTypeList.Update: return (int)ppms_ServiceTransactionType.Update;
                case (int)TransactionTypeList.DeactivateProvider: return (int)ppms_ServiceTransactionType.DeactivateProvider;
                case (int)TransactionTypeList.DeactivateRelationship: return (int)ppms_ServiceTransactionType.DeactivateRelationship;
            }

            throw new PpmsServiceException("Error writing Batch record for Schema Validation. Invalid TransactionType.");
        }

        private async Task InsertEntity(Entity entity)
        {
            using (var proxy = await _ppmsContextHelper.GetOrganizationServiceProxyAsync())
            {
                proxy.Create(entity);
            }
        }

        private async Task UpdateEntity(Entity entity)
        {
            using (var proxy = await _ppmsContextHelper.GetOrganizationServiceProxyAsync())
            {
                proxy.Update(entity);
            }
        }

        private async Task UpdateProvider(Account provider)
        {
            using (var context = await _ppmsContextHelper.GetContextAsync())
            {
                if (!context.IsAttached(provider))
                {
                    context.Attach(provider);
                }

                context.SaveChanges();
            }
        }

        private async Task DeleteEntity(IList<SetStateRequest> requests)
        {
            using (var proxy = await _ppmsContextHelper.GetOrganizationServiceProxyAsync())
            {
                foreach (var request in requests)
                {
                    proxy.Execute(request);
                }
            }
        }

        private async Task DeleteAccounts(IList<MapperResultDetail> accounts)
        {
            using (var proxy = await _ppmsContextHelper.GetOrganizationServiceProxyAsync())
            {
                foreach (var account in accounts)
                {
                    foreach (var request in account.Requests)
                    {
                        proxy.Execute(request);
                    }
                }
            }
        }

        private static async Task AddRelatedEntitiesToContext(OrganizationWebProxyClient context, Entity parent)
        {
            foreach (var entity in parent.RelatedEntities)
            {
                if (entity.Value == null || !entity.Value.Entities.Any()) continue;

                context.Associate(parent.LogicalName, parent.Id, entity.Key, ToEntityReferenceCollection(entity.Value));
                foreach (var item in entity.Value.Entities)
                {
                    await AddRelatedEntitiesToContext(context, item);
                }
            }
        }

        private static void AddRelatedEntitiesToContext(PpmsContext context, Entity parent)
        {
            foreach (var entity in parent.RelatedEntities)
            {
                if (entity.Value == null || !entity.Value.Entities.Any()) continue;

                foreach (var item in entity.Value.Entities)
                {
                    if (!context.IsAttached(item))
                    {
                        context.Attach(item);
                        context.AttachLink(parent, entity.Key, item);
                    }
                    AddRelatedEntitiesToContext(context, item);
                }
            }
        }

        private static EntityReferenceCollection ToEntityReferenceCollection(EntityCollection entityList)
        {
            var result = new EntityReferenceCollection();

            if (entityList == null || !entityList.Entities.Any()) return result;

            foreach (var item in entityList.Entities)
            {
                result.Add(item.ToEntityReference());
            }

            return result;
        }
    }
}